﻿#region << Using Directives >>
using System;
#endregion

namespace Volpe.Cafe.Generic
{
    /// <summary>
    /// Represents an object that stores compliance modeling data for each vehicle class.
    /// </summary>
    [Serializable]
    public class VCValue<T> : ICloneable
    {

        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="VCValue{T}"/> class.
        /// </summary>
        public VCValue()
        {
            this.Items = new T[Classes.Length];
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="VCValue{T}"/> class using the specified values.
        /// </summary>
        public VCValue(T ldv, T ldt12a, T ldt2b3)
        {
            this.Items = new T[] { ldv, ldt12a, ldt2b3 };
        }

        #endregion

        #region /*** Methods ***/

        #region /* Overloaded Operators */

        /// <summary>
        /// Determines whether the specified <see cref="VCValue{T}"/> values are equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="VCValue{T}"/> values are equal; false, otherwise.</returns>
        public static bool operator ==(VCValue<T> value1, VCValue<T> value2)
        {
            return Equals(value1, value2);
        }
        /// <summary>
        /// Determines whether the specified <see cref="VCValue{T}"/> values are not equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="VCValue{T}"/> values are not equal; false, otherwise.</returns>
        public static bool operator !=(VCValue<T> value1, VCValue<T> value2)
        {
            return !Equals(value1, value2);
        }

        #endregion

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a shallow copy of the current <see cref="VCValue{T}"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="VCValue{T}"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a shallow copy of the current <see cref="VCValue{T}"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="VCValue{T}"/>.</returns>
        public VCValue<T> Clone()
        {
            VCValue<T> value = new VCValue<T>();
            this.CopyTo(value);
            return value;
        }
        /// <summary>
        /// Copies the members of the current <see cref="VCValue{T}"/> instance into the specified value.
        /// </summary>
        /// <param name="value">The value where to copy the members of the current instance.</param>
        protected void CopyTo(VCValue<T> value)
        {
            for (int i = 0; i < this.Items.Length; i++)
            {
                value.Items[i] = this.Items[i];
            }
        }

        #endregion

        #region /* Overriden from object */

        /// <summary>
        /// Returns the string representation of this <see cref="VCValue{T}"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="VCValue{T}"/> instance.</returns>
        public override string ToString()
        {
            string s = string.Empty;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (i > 0) { s += ", "; }
                s += (Names[i] + "=" + this.Items[i].ToString());
            }
            return "{" + s + "}";
        }

        /// <summary>
        /// Serves as a hash function for a particular type, suitable for use in hashing algorithms and data structures like a hash table.
        /// </summary>
        /// <returns>A hash code for the current <see cref="VCValue{T}"/>.</returns>
        public override int GetHashCode()
        {
            int hash = 0;
            for (int i = 0; i < this.Items.Length; i++)
            {
                hash = hash ^ this.Items[i].GetHashCode();
            }
            return hash;
        }

        /// <summary>
        /// Determines whether the specified <see cref="Object"/> is equal to the current <see cref="VCValue{T}"/> value.
        /// </summary>
        /// <param name="obj">The <see cref="Object"/> to compare with the current <see cref="VCValue{T}"/> value.</param>
        /// <returns>true, if the specified <see cref="Object"/> is equal to the current <see cref="VCValue{T}"/> value; false, otherwise.</returns>
        public override bool Equals(object obj)
        {
            return (obj is VCValue<T>) ? this.Equals((VCValue<T>)obj) : base.Equals(obj);
        }
        /// <summary>
        /// Determines whether the specified <see cref="VCValue{T}"/> value is equal to the current <see cref="VCValue{T}"/> value.
        /// </summary>
        /// <param name="value">The <see cref="VCValue{T}"/> value to compare with the current <see cref="VCValue{T}"/> value.</param>
        /// <returns>true, if the specified <see cref="VCValue{T}"/> value is equal to the current <see cref="VCValue{T}"/> value;
        ///   false, otherwise.</returns>
        public bool Equals(VCValue<T> value)
        {
            return Equals(this, value);
        }
        /// <summary>
        /// Determines whether the specified <see cref="VCValue{T}"/> values are equal.
        /// </summary>
        /// <param name="value1">The first <see cref="VCValue{T}"/> value to compare.</param>
        /// <param name="value2">The second <see cref="VCValue{T}"/> value to compare.</param>
        /// <returns>true, if the specified <see cref="VCValue{T}"/> values are equal; false, otherwise.</returns>
        public static bool Equals(VCValue<T> value1, VCValue<T> value2)
        {
            if (object.ReferenceEquals(value1, null) && object.ReferenceEquals(value2, null)) { return true; }
            if (object.ReferenceEquals(value1, null) || object.ReferenceEquals(value2, null)) { return false; }
            //
            for (int i = 0; i < value1.Items.Length; i++)
            {
                if (!value1.Items[i].Equals(value2.Items[i])) { return false; }
            }
            return true;
        }

        #endregion

        /// <summary>
        /// Resets all members of this <see cref="VCValue{T}"/> instance to their default values (0, false, or null).
        /// </summary>
        public void Clear()
        {
            for (int i = 0; i < this.Items.Length; i++)
            {
                this.Items[i] = default(T);
            }
        }

        /// <summary>
        /// Returns the index corresponding to the specified <see cref="VehicleClass"/>.
        /// </summary>
        /// <param name="vehClass"></param>
        /// <returns>The index corresponding to the specified <see cref="VehicleClass"/>.</returns>
        protected int GetIndex(VehicleClass vehClass)
        {
            //return (int)vehClass - 1;
            switch (vehClass)
            {
                case VehicleClass.LDV   : return  0;
                //
                case VehicleClass.LDT1  :
                case VehicleClass.LDT2a :
                case VehicleClass.LDT12a: return  1;
                //
                case VehicleClass.LDT2b :
                case VehicleClass.LDT3  :
                case VehicleClass.LDT2b3: return  2;
                //
                default:                  return -1;
            }
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets or sets the compliance modeling data value based on the given vehicle class.</summary>
        /// <exception cref="System.IndexOutOfRangeException">The value specified does not represent a valid vehicle class.</exception>
        public T this[VehicleClass vehClass]
        {
            get { return this.Items[this.GetIndex(vehClass)]; }
            set { this.Items[this.GetIndex(vehClass)] = value; }
        }

        /// <summary>Gets an array of compliance modeling data specified by <see cref="VehicleClass"/>.</summary>
        public T[] Items { get; private set; }

        #endregion

        #region /*** Variables ***/

        /// <summary>Specifies the <see cref="VehicleClass"/>es supported by this object. This field is read-only.</summary>
        public static readonly VehicleClass[] Classes = new VehicleClass[] { VehicleClass.LDV,
                                                                             VehicleClass.LDT12a,
                                                                             VehicleClass.LDT2b3 };
        /// <summary>Provides the friendly name for each of the supported <see cref="VehicleClass"/>es. This field is read-only.</summary>
        public static readonly string[] Names = new string[] { "LDV",
                                                               "LDT12a",
                                                               "LDT2b3" };

        #endregion

    }
}
